#ifndef __CXMLDocument__
#define __CXMLDocument__

#include "CTextStream.hpp"
#include "CXMLNode.hpp"
using Exponent::IO::CTextStream;
using Exponent::IO::CXMLNode;

namespace Exponent
{
	namespace IO
	{
		/**
		 * @class CXMLDocument CXMLDocument.hpp
		 * @brief Represents an xml document tree
		 *
		 * XML is at its basic level a tree of nodes. We follow this paradigm. An XML document will be written or read from a root node\n
		 * that is the root of the tree. Each node may have n children who may in turn have n children.\n
		 * Each element of the tree has a specific number (>= 0) of attributes\n
		 * All attributes, children and names etc are represented by strings, but there are various ways in the API of converting between\n
		 * strings and other objects, values etc\n
		 * Here is an example of how to write a file
		 * @code
		 * // Can i just say, yes this entire example is about the tv show
		 * // The shield... possible the finest show ever made :)
		 * // First create the node that you are gonna fill
		 * CXMLNode *root = new CXMLNode("root", NULL);
		 *
		 * // Now add some attributes and children to it
		 * CXMLNode *strikeTream = new CXMLNode("strike_team", root);
		 * strikeTeam->addAttribute("number_of_members", "4");
		 *
		 * // Now add children
		 * CXMLNode *vic   = new CXMLNode("vic",   strikeTeam);
		 * CXMLNode *lemm  = new CXMLNode("lemm",  strikeTeam);
		 * CXMLNode *ronny = new CXMLNode("ronny", strikeTeam);
		 * CXMLNode *shane = new CXMLNode("shane", strikeTeam);
		 *
		 * // Set some attributes
		 * vic->addAttribute("description",  "hard ass");
		 * lemm->addAttribute("description", "The good guy");
		 * ronny->addAttribute("description", "Who?");
		 * shane->addAttribute("description", "Scumm bag");
		 *
		 * // Now add them to the strike team node
		 * strikeTeam->addChildNode(vic);
		 * strikeTeam->addChildNode(lemm);
		 * strikeTeam->addChildNode(ronny);
		 * strikeTeam->addChildNode(shane);
		 *
		 * // Add to the root -> notice that this reference counts all the objects, we can just ignore those pointers now...
		 * root->addChildNode(strikeTeam);
		 *
		 * // Now set and write
		 * CXMLDocument document;
		 * document.setRootNode(root);
		 * if (!document.writeFile("Path/To/MyFile.xml"))
		 * {
		 *     cout << "Failed to write document" << endl;
		 * }
		 * @endcode
		 * \n
		 * Here is an example of how to read a file:\n
		 * @code
		 * // We are going to read the document, then we have the node available to process
		 * CXMLDocument document;
		 * if (!document.readFile("Path/To/MyFile.xml"))
		 * {
		 *     cout << "Failed to read document" << endl;
		 * }
		 * else
		 * {
		 * 		// Store the root node
		 * 		const CXMLNode *root = document.getRootNode();
		 *
		 *		// Check its valid
		 *		if (root == NULL) return;
		 *
		 *      // Now we can process with the node
		 *      ...
		 * }
		 * @endcode
		 *
		 * @see CXMLAttribute
		 * @see CXMLNode
		 *
		 * @date 17/08/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note This is a very basic implementation of XML reading and writing.\n
		 * It is intended as an opening in to the world of XML to be updated at a later date\n
		 * Currently not supported are multiline comments nor seperate DTD nor unicode\n
		 * Comliance with XML documents written by other sources may be a bit buggy, please make sure you check your code ;)
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CXMLDocument.hpp,v 1.5 2007/02/08 21:06:44 paul Exp $
		 */
		class CXMLDocument : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 */
			CXMLDocument();

			/**
			 * Destruction
			 */
			virtual ~CXMLDocument();

//	===========================================================================

			/**
			 * Read an entire file and create the XML object tree
			 * @param filename The name of the file
			 * @retval bool True if read properly, false otherwise
			 */
			bool readFile(const CSystemString &filename);

			/**
			 * Write the tree as stored in root node to disk file
			 * @param filename The name of the file to write to
			 * @retval bool True if written properly, false otherwise
			 */
			bool writeFile(const CSystemString &filename);

//	===========================================================================

			/**
			 * Get the root node
			 * @retval const CXMLNode* The root node, possible NULL
			 */
			const CXMLNode *getRootNode() const;

			/**
			 * Get the root node
			 * @retval const CXMLNode* The root node, possible NULL
			 */
			CXMLNode *getMutableRootNode();

			/**
			 * Set the root node
			 * @param root The root node
			 */
			void setRootNode(CXMLNode *root);

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * Read the DTD
			 * @param stream The textt stream to write from
			 * @retval bool True if read properly, false otherwise
			 */
			bool readDTD(CTextStream &stream);

			/**
			 * Process a given string
			 * @param string The string to read
			 * @param stream The text stream to read from as necessary
			 * @retval bool True if read properly, false otherwise
			 */
			bool processString(CString &string, CTextStream &stream);

//	===========================================================================

			/**
			 * Output the DTD header - <?xml version=\"1.0\" encoding=\"UTF-8\" \?>
			 * @param stream The textt stream to write to
			 */
			void outputDTD(CTextStream &stream);

			/**
			 * Indent the output
			 * @param stream The text stream to indent
			 */
			void indentOutput(CTextStream &stream);

			/**
			 * Output a node
			 * @param node The xml node to output
			 * @param stream The text stream to use
			 */
			void outputNode(CXMLNode *node, CTextStream &stream);

//	===========================================================================

			/**
			 * Replace output string characters that are illegal with legal versions of them
			 * @param string The string to check and replace
			 */
			void replaceIllegalCharacters(CString &string);

			/**
			 * Replace input string character that are legal with illegal versions of them
			 * @param string The string to check and replace
			 */
			void replaceEscapeCharacters(CString &string);

//	===========================================================================

			CXMLNode *m_readNode;									/**< Traversal node */
			CXMLNode *m_rootNode;									/**< Root node */
			long m_indentLevel;										/**< How many levels of indent are we at */
		};
	}
}

#endif		// End of CXMLDocument.hpp